% ukf_endo_ms_lik_2nd_kim_nelson_create.m
% 
% unscented kalman filter file for computing likelihood-based effects of
% setting a shock to zero
% 
% Estimating Macroeconomic Models of Financial Crises: An Endogenous Regime-Switching Approach*
%   Gianluca Benigno, Andrew Foerster, Christopher Otrok, Alessandro Rebucci
% 
% Updated July 2024 
% % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % %
% 
% INPUTS
%   xss         = (nx x 1) vector, steady state of predetermined variables
%   yss         = (ny x 1) vector, steady state of nonpredetermined variables
%   Pss         = (ns x ns) matrix, steady state transition matrix
%   a           = (NX x ns) matrix, intial set of estimation states for
%                   each regime
%   P           = (NX x NX x ns) array, intial covariance matrix for states
%                   in each regime
%   PAI00       = (ns x 1) matrix, initial probability distribution across
%                   regimes
%   ggammas     = (2 x 2) matrix, with elements ggamma_{ii-1,jj-1} for
%                   transition matrix
%   data_y      = (NY x T) matrix of data for estimating the likelihood
%   D           = (NY x NX x ns) array, observation equation in each regime
%   E           = (NY x NY x ns) array, errors in obs equation
%                   Y(t) = D[s] *X(t) + E[s]*U(t)
%   H           = (nx x nx+ne+1 x ns) array, 1st order predetermined
%   H2          = (nx x (nx+ne+1)^2 x ns) array, 2st order predetermined
%   G           = (ny x nx+ne+1 x ns) array, 1st order nonpredetermined
%   G2          = (ny x (nx+ne+1)^2 x ns) array, 2st order nonpredetermined
%   kf_filtering_level = scalar determining filter output, with 
%           if 0, outputs the updated variables
%           if 1, outputs the the smoothed variables
% 
% OUTPUTS
%   loglike     = scalar, value of the log-likelihood
%   Incr        = (T x 1) vector, marginal likelihood
%   retcode     = boolean, 0 if error in producing likelihood
%   Filters     = structure, elements depending on 'kf_filtering_level'
%       PAI     = (ns x T) matrix, predicted probabilities across regimes
%       Q       = (ns x ns x T) array, predicted transition matrix
%     	a       = (NX x T x ns x ns) array, predicted state variables
%       P       = (NX x NX x T x ns x ns) array, predicted covariance matrix
%       att     = (NX x T x ns) array, filtered state variables
%       Ptt     = (NX x NX x T x ns) array, filtered covariance matrix
%       PAItt   = (ns x T) matrix, filtered probabilities across regimes
%       atT     = (NX x T x ns) array, smoothed state variables
%       PAItT   = (ns x T) matrix, smoothed probabilities across regimes

% 
% % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % % %
function [loglik,loglik_counter,retcode,Filters] ...
        = ukf_endo_ms_2nd_kim_nelson_create_SV_counter(GLOBALS,xss,yss,Pss,a,P,PAI00,ggammas,Psigma,data_y,D,E,H,H2,G,G2,V,kf_filtering_level,counter_t,counter_e)


% -- Initial Conditions -- %
att     = a;        % initial state variables
Ptt     = P;        % initial covariance matrix of the states in each regime
PAItt   = PAI00;	% initial probability distributions of regimes
Q       = Pss;      % transition matrix


% -- Counterfactual Initial Conditions -- %
att_counter  	= a;        % initial state variables
Ptt_counter 	= Ptt;      % initial covariance matrix of the states in each regime
PAItt_counter   = PAI00;	% initial probability distributions of regimes
Q_counter       = Pss;      % transition matrix

% -- System Dimensions -- %
[p,smpl]    = size(data_y);
m           = size(a,1);    % number of state variables in estimation
h           = length(Pss);  % number of regimes
ne          = length(V);    % number of shocks   
nx          = length(xss);  % number of predetermined varialbes
ny          = length(yss);  % number of nonpredetermined variables

% -- Initialization of Matrices -- %
loglik      = [];               % value of log-likelihood
Incr        = zeros(smpl,1);
twopi_p_dF  = zeros(1,h);
K           = zeros(m,p,h);
retcode     = 1;

% -- Initialization of Counterfactual Matrices -- %
Incr_counter        = zeros(smpl,1);    
twopi_p_dF_counter  = zeros(1,h);
K_counter           = zeros(m,p,h);



% --  Weights for Sigma Points -- %
mm              = m + ne;
k               = 3 - mm;
wm              = 1/(2*(mm+k))*ones(1,2*mm+1);
wc              = 1/(2*(mm+k))*ones(1,2*mm+1);
wm(1)           = k/(mm+k);
wc(1)           = k/(mm+k);
sqrt_m_plus_k   = sqrt(mm+k);

% -- Transition Matrix -- %
ggamma00 = ggammas(1,1);
ggamma01 = ggammas(1,2);
ggamma10 = ggammas(2,1);
ggamma11 = ggammas(2,2);

% -- Storage -- %
a                   = zeros(m,h,h);         % predicted state value
PS                  = zeros(m,m,h,h);       % predicted covariance matrix
PAI01y              = zeros(h,h);           % bridge for filtered probability
SG                  = zeros(m,m,smpl,h,h);  % smoothing gain
iPvv                = zeros(p,p,h,h);       % inverse covariance matrix for observables
Ptt_raw             = zeros(m,m,h,h);       % filtered covariance matrix, conditional on regimes
att_raw             = zeros(m,h,h);         % filtered state variables, conditional on regimes
f01                 = zeros(h,h);           % marginal likelihood for each regime
v                   = zeros(p,h,h);         % prediction error for observables, conditional on each regime
a_plus_temp         = zeros(m,2*mm+1);      % temporary predicted state variables
x_plus              = zeros(m,2*mm+1);      % predicted sigma points
P_plus_temp         = zeros(m,m,2*mm+1);    % temporary predicted covariance matrix
C_t_tp1_temp        = zeros(m,m,2*mm+1);    % temporary matrix for calculating smoothed values
att_temp            = zeros(m,h,h);         % temporary matrix for filtered state variables
Ptt_temp            = zeros(m,m,h,h);       % temporary matrix for filtered covariance matrix
Filters.PAI         = zeros(h,smpl);        % predicted probabilities across regimes
Filters.Q           = zeros(h,h,smpl);      % predicted transition matrix
Filters.a           = zeros(m,smpl,h,h);    % predicted state variables
Filters.P           = zeros(m,m,smpl,h,h);  % predicted covariance matrix
Filters.PAItt       = zeros(h,smpl);        % filtered probabilities across regimes
Filters.att         = zeros(m,smpl,h);      % filtered state variables
Filters.Ptt         = zeros(m,m,smpl,h);    % filtered covariance matrix
Filters.atT         = zeros(m,h,smpl);      % smoothed state variables
Filters.PAItT       = zeros(h,smpl);        % smoothed probabilities across regimes
Filters.atT_temp	= zeros(m,1,smpl,h,h);  % temporary matrix for calculating smoothed state variables
pai_0_1             = zeros(h,h,smpl);      % temporary matrix for calculating smoothed probabilities

% -- Storage: Counterfactuals --%
a_counter                   = zeros(m,h,h);         % predicted state value
PS_counter                  = zeros(m,m,h,h);       % predicted covariance matrix
PAI01y_counter              = zeros(h,h);           % bridge for filtered probability
SG_counter                  = zeros(m,m,smpl,h,h);  % smoothing gain
iPvv_counter                = zeros(p,p,h,h);       % inverse covariance matrix for observables
Ptt_raw_counter             = zeros(m,m,h,h);       % filtered covariance matrix, conditional on regimes
att_raw_counter             = zeros(m,h,h);         % filtered state variables, conditional on regimes
f01_counter                 = zeros(h,h);           % marginal likelihood for each regime
v_counter                   = zeros(p,h,h);         % prediction error for observables, conditional on each regime
a_plus_temp_counter         = zeros(m,2*mm+1);      % temporary predicted state variables
x_plus_counter              = zeros(m,2*mm+1);      % predicted sigma points
P_plus_temp_counter         = zeros(m,m,2*mm+1);    % temporary predicted covariance matrix
C_t_tp1_temp_counter        = zeros(m,m,2*mm+1);    % temporary matrix for calculating smoothed values
att_temp_counter            = zeros(m,h,h);         % temporary matrix for filtered state variables
Ptt_temp_counter            = zeros(m,m,h,h);       % temporary matrix for filtered covariance matrix
Filters.PAI_counter         = zeros(h,smpl);        % predicted probabilities across regimes
Filters.Q_counter           = zeros(h,h,smpl);      % predicted transition matrix
Filters.a_counter           = zeros(m,smpl,h,h);    % predicted state variables
Filters.P_counter           = zeros(m,m,smpl,h,h);  % predicted covariance matrix
Filters.PAItt_counter       = zeros(h,smpl);        % filtered probabilities across regimes
Filters.att_counter         = zeros(m,smpl,h);      % filtered state variables
Filters.Ptt_counter         = zeros(m,m,smpl,h);    % filtered covariance matrix
Filters.atT_counter         = zeros(m,h,smpl);      % smoothed state variables
Filters.PAItT_counter       = zeros(h,smpl);        % smoothed probabilities across regimes
Filters.atT_temp_counter    = zeros(m,1,smpl,h,h);  % temporary matrix for calculating smoothed state variables
pai_0_1_counter             = zeros(h,h,smpl);      % temporary matrix for calculating smoothed probabilities


% -- Counterfactual Objects -- %

if counter_e == 1
    counter_select    = 49; 
elseif counter_e == 2
    counter_select    = 50; 
elseif counter_e == 3
    counter_select    = 51; 
elseif counter_e == 4
    counter_select    = 52; 
elseif counter_e == 5
    counter_select    = 53; 
elseif counter_e == 6
    counter_select    = 54;
end
V_counter = 0*V;

% -- Start Running Filters -- %
for t = 1:smpl      % <-- t=0; while t<smpl,t=t+1;

    % -- Observables for date t -- %
    y = data_y(:,t);
    

    % -- Endogenous Probabilities, Conditional on Time t Information -- %
    if t == 1
        % -- Initial Transitions Based on Steady State -- %
        Filters.PAI(:,t)    = PAI00;
        Filters.Q(:,:,t)    = Q;
        
        Filters.PAI_counter(:,t)    = PAI00;
        Filters.Q_counter(:,:,t)    = Q;
    else
        % -- Weighted Average of Lambda and B* -- %
        llambda = PAItt(1)*att(2*GLOBALS.nx+GLOBALS.var_ind.y.llambda,1) ...
            + PAItt(2)*att(2*GLOBALS.nx+GLOBALS.var_ind.y.llambda,2);    
        bstar   = PAItt(1)*att(2*GLOBALS.nx+GLOBALS.var_ind.y.bstar,1) ...
            + PAItt(2)*att(2*GLOBALS.nx+GLOBALS.var_ind.y.bstar,2);
        
        % -- Weighted Average of Lambda and B* -- %
        llambda_counter = PAItt_counter(1)*att_counter(2*GLOBALS.nx+GLOBALS.var_ind.y.llambda,1) ...
            + PAItt_counter(2)*att_counter(2*GLOBALS.nx+GLOBALS.var_ind.y.llambda,2);    
        bstar_counter   = PAItt_counter(1)*att_counter(2*GLOBALS.nx+GLOBALS.var_ind.y.bstar,1) ...
            + PAItt_counter(2)*att_counter(2*GLOBALS.nx+GLOBALS.var_ind.y.bstar,2);

        if h>1
            % -- Set Transition Matrix -- %
            Pc = [1-exp(ggamma00-ggamma01*bstar)/(1+exp(ggamma00-ggamma01*bstar))     exp(ggamma00-ggamma01*bstar)/(1+exp(ggamma00-ggamma01*bstar));...
            exp(ggamma10-ggamma11*llambda)/(1+exp(ggamma10-ggamma11*llambda))    1-exp(ggamma10-ggamma11*llambda)/(1+exp(ggamma10-ggamma11*llambda))];
            Q = kron(Pc,Psigma);

            % -- Probabilities Predictions -- %
            PAI=Q'*PAItt;
            
            % -- Set Transition Matrix -- %
            Pc_counter = [1-exp(ggamma00-ggamma01*bstar_counter)/(1+exp(ggamma00-ggamma01*bstar_counter))     exp(ggamma00-ggamma01*bstar_counter)/(1+exp(ggamma00-ggamma01*bstar_counter));...
            exp(ggamma10-ggamma11*llambda_counter)/(1+exp(ggamma10-ggamma11*llambda_counter))    1-exp(ggamma10-ggamma11*llambda_counter)/(1+exp(ggamma10-ggamma11*llambda_counter))];
            Q_counter = kron(Pc_counter,Psigma);
            
            % -- Probabilities Predictions -- %
            PAI_counter=Q_counter'*PAItt_counter;
        end
        

        % -- Save Transition Matrix and Probabilities -- %
        Filters.PAI(:,t)    = PAI;
        Filters.Q(:,:,t)    = Q;
        
        % -- Save Transition Matrix and Probabilities -- %
        Filters.PAI_counter(:,t)    = PAI_counter;
        Filters.Q_counter(:,:,t)    = Q_counter;
    end
    
    % -- Looping Over Current Regime -- %
    for st=1:h

        % -- Combine State and Shock Vectors -- %
        aug_temp_a = [att(:,st);zeros(ne,1)];
        aug_temp_P = blkdiag(Ptt(:,:,st),V);
        aug_temp_a_counter = [att_counter(:,st);zeros(ne,1)];
        aug_temp_P_counter = blkdiag(Ptt_counter(:,:,st),V);
        
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        if t == counter_t
            aug_temp_a_counter = [att_counter(:,st);zeros(ne,1)];
            aug_temp_P_counter = blkdiag(Ptt_counter(:,:,st),V_counter);
        end
        %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        
        % -- Stop if Invalid State Vector -- %
        if sum(isnan(aug_temp_a)) ~= 0
            retcode = 0;
            return;
        end
        % -- Get Sigma Points -- %
        [xtt,chkcode1] = sigma_points(aug_temp_a,aug_temp_P,sqrt_m_plus_k);
        [xtt_counter,chkcode2] = sigma_points(aug_temp_a_counter,aug_temp_P_counter,sqrt_m_plus_k);
        if chkcode1 == 999 || chkcode2 == 999
            retcode = 0;
            return;
        end

        % -- Looping Over Future Regime -- %
        for splus=1:h

            
            % -- Run Sigma Points for States through State Equation -- %
            %for jj=1:2*mm+1
            %    % use state equation
            %    x_plus(:,jj)        = ff2(ny,nx,ne,yss,xss,H,G,H2,G2,xtt(1:m,jj),xtt(m+1:m+ne,jj)',splus);
            %    x_plus_counter(:,jj) = ff2(ny,nx,ne,yss,xss,H,G,H2,G2,xtt_counter(1:m,jj),xtt_counter(m+1:m+ne,jj)',splus);
                
            %    % assign weights
            %    a_plus_temp(:,jj)           = wm(jj)*x_plus(:,jj);
            %    a_plus_temp_counter(:,jj)   = wm(jj)*x_plus_counter(:,jj);
            %end
            % take average
            %a_plus1          = sum(a_plus_temp,2);
            %a_plus_counter1  = sum(a_plus_temp_counter,2);



            x_plus = ff2_noloop_mex(ny,nx,ne,yss,xss,H(:,:,splus),G(:,:,splus),H2(:,:,splus),G2(:,:,splus),xtt(1:m,:),xtt(m+1:m+ne,:),[m;mm]);                      
            a_plus = x_plus*wm';
            x_plus_counter = ff2_noloop_mex(ny,nx,ne,yss,xss,H(:,:,splus),G(:,:,splus),H2(:,:,splus),G2(:,:,splus),xtt_counter(1:m,:),xtt_counter(m+1:m+ne,:),[m;mm]);                      
            a_plus_counter = x_plus_counter*wm';

            
            % -- Sigma Points for Covariance Matrix -- %        
            for jj = 1:2*mm+1
                % deviations of each point from mean
                x_a = x_plus(:,jj) - a_plus;
                x_a_counter = x_plus_counter(:,jj) - a_plus_counter;
                
                % weighted quadratic form                
                P_plus_temp(:,:,jj) = wc(jj)*(x_a*x_a.');
                P_plus_temp_counter(:,:,jj) = wc(jj)*(x_a_counter*x_a_counter.');
                
                % 
                if kf_filtering_level>0 && jj<=2*mm+1
                    C_t_tp1_temp(:,:,jj) = wc(jj)*(xtt(1:m,jj)-att(:,st))*x_a.';
                    C_t_tp1_temp_counter(:,:,jj) = wc(jj)*(xtt_counter(1:m,jj)-att_counter(:,st))*x_a_counter.';
                end
            end
            % take average 
            P_plus                  = sum(P_plus_temp,3);
            C_t_tp1                 = sum(C_t_tp1_temp,3);
            P_plus_counter          = sum(P_plus_temp_counter,3);
            C_t_tp1_counter         = sum(C_t_tp1_temp_counter,3);
            
            % -- Save States and Covariance Matrix -- %
            a(:,st,splus)           = a_plus;
            a_counter(:,st,splus)   = a_plus_counter;
            Filters.a(:,t,st,splus) = a(:,st,splus);
            Filters.a_counter(:,t,st,splus) = a_counter(:,st,splus);
            
            % -- Symmetrization/Projection Step -- %
            % stop if covariance matrix is NaN
            if isnan(sum(sum(P_plus)))
                retcode = 0;
                return;
            end
            
            % if covariance matrix is not symmetric, project to new one
            [~,chk] = chol(P_plus);
            if chk>0
                P_plus = project(P_plus,0);
            end
            [~,chk] = chol(P_plus_counter);
            if chk>0
                P_plus_counter = project(P_plus_counter,0);
            end
            
            % -- Save Covariance Matrix -- %
            PS(:,:,st,splus)            = P_plus;
            Filters.P(:,:,t,st,splus)   = PS(:,:,st,splus);
            PS_counter(:,:,st,splus)          = P_plus_counter;
            Filters.P_counter(:,:,t,st,splus) = PS_counter(:,:,st,splus);
            
            
            % -- Compute Smoothing Gain, Store for Later Use -- %
            if kf_filtering_level>0
                SG(:,:,t,st,splus) = C_t_tp1*pinv(project(P_plus,1));
                SG_counter(:,:,t,st,splus) = C_t_tp1_counter*pinv(project(P_plus_counter,1));
            end
            
            
            % -- Forecast Observables -- %
            yf = D(:,:,splus)*a(:,st,splus); %<-- yf=D*a{st};
            yf_counter = D(:,:,splus)*a_counter(:,st,splus); %<-- yf=D*a{st};
            
            % -- Forecast Errors -- %
            v(:,st,splus)=y-yf;
            v_counter(:,st,splus)=y-yf_counter;

            % -- Forecast Variance -- %
            Pvv     = D(:,:,splus)*PS(:,:,st,splus)*D(:,:,splus)'+E(:,:,splus);
            Pav     = PS(:,:,st,splus)*D(:,:,splus)'; % PZt=<-- P{st}*D';
            detF    = det(Pvv);
            iPvv(:,:,st,splus) = Pvv\eye(p);
            
            Pvv_counter     = D(:,:,splus)*PS_counter(:,:,st,splus)*D(:,:,splus)'+E(:,:,splus);
            Pav_counter     = PS_counter(:,:,st,splus)*D(:,:,splus)'; % PZt=<-- P{st}*D';
            detF_counter    = det(Pvv_counter);
            iPvv_counter(:,:,st,splus) = Pvv_counter\eye(p);

            % -- Kalman Gain for Update -- %
            K(:,:,splus) = Pav*iPvv(:,:,st,splus); % K=PZt/F{st};
            K_counter(:,:,splus) = Pav_counter*iPvv_counter(:,:,st,splus); % K=PZt/F{st};

            % -- State covariance update (Ptt=P-P*D*inv(F)*D'*P) -- %
            Ptt_raw(:,:,st,splus)   = PS(:,:,st,splus)-K(:,:,splus)*Pav.';%<---P{st}=P{st}-K(:,occur,st)*P{st}(obsOccur,:);
            twopi_p_dF(splus)       = (2*pi)^p*detF;
            
            Ptt_raw_counter(:,:,st,splus)   = PS_counter(:,:,st,splus)-K_counter(:,:,splus)*Pav_counter.';%<---P{st}=P{st}-K(:,occur,st)*P{st}(obsOccur,:);
            twopi_p_dF_counter(splus)       = (2*pi)^p*detF_counter;
            if t == counter_t
                temp = 3:8;
                temp = temp(2+counter_e~=temp);
                Ptt_raw_counter(temp,:,st,splus) = 0;
                Ptt_raw_counter(:,temp,st,splus) = 0;
                Ptt_raw_counter(2*nx+2*ny+1:2*nx+2*ny+ne,:,st,splus) = 0;
                Ptt_raw_counter(:,2*nx+2*ny+1:2*nx+2*ny+ne,st,splus) = 0;
                %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
            end
            
            % -- State Update (att=a+K*v) -- %
            att_raw(:,st,splus) = a(:,st,splus)+K(:,:,splus)*v(:,st,splus);
            att_raw_counter(:,st,splus) = a_counter(:,st,splus)+K_counter(:,:,splus)*v_counter(:,st,splus);            
            if t == counter_t
                temp = 3:8;
                temp = temp(2+counter_e~=temp);                
                att_raw_counter(temp,st,splus) = att_raw(temp,st,splus);
                att_raw_counter(2*nx+2*ny+1:2*nx+2*ny+ne,st,splus) = att_raw(2*nx+2*ny+1:2*nx+2*ny+ne,st,splus);
                att_raw_counter(counter_select,st,splus)=0;
                %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
            end
            
            % -- Likelihood Contribution -- %
            f01(st,splus)       = (twopi_p_dF(splus)*exp(v(:,st,splus)'*iPvv(:,:,st,splus)*v(:,st,splus)))^(-0.5) + 1.0e-99;
            PAI01y(st,splus)    = Q(st,splus)*PAItt(st)*f01(st,splus);
            f01_counter(st,splus)       = (twopi_p_dF_counter(splus)*exp(v_counter(:,st,splus)'*iPvv_counter(:,:,st,splus)*v_counter(:,st,splus)))^(-0.5) + 1.0e-99;
            PAI01y_counter(st,splus)    = Q_counter(st,splus)*PAItt_counter(st)*f01_counter(st,splus);
        end
    end
    
    % -- Sump Likelihood across Regimes -- %
    likt=sum(sum(PAI01y));
    likt_counter=sum(sum(PAI01y_counter));
    
    % -- Probability updates -- %
    PAI01_tt=PAI01y/likt;
    PAI01_tt_counter=PAI01y_counter/likt_counter;
    
    % -- Likelihood Computation -- %
    Incr(t)=log(likt);
    PAItt=sum(PAI01_tt,1)';
    Incr_counter(t)=log(likt_counter);
    PAItt_counter=sum(PAI01_tt_counter,1)';

    % -- Save Probabilities -- %
    Filters.PAItt(:,t)=PAItt;
    Filters.PAItt_counter(:,t)=PAItt_counter;
    
    % -- Kim-Nelson Collapsing Step for States -- %
    for splus=1:h
        for st=1:h
            pai_st_splus                    = PAI01_tt(st,splus)/PAItt(splus);
            att_temp(:,splus,st)            = pai_st_splus*att_raw(:,st,splus);
            pai_st_splus_counter            = PAI01_tt_counter(st,splus)/PAItt_counter(splus);
            att_temp_counter(:,splus,st)    = pai_st_splus_counter*att_raw_counter(:,st,splus);
        end
        att(:,splus)                    = sum(att_temp(:,splus,:),3);
        att_counter(:,splus)            = sum(att_temp_counter(:,splus,:),3);
        if t == counter_t
            temp = 3:8;
            temp = temp(2+counter_e~=temp);  
            att_counter(temp,splus) = att(temp,splus);
            att_counter(2*nx+2*ny+1:2*nx+2*ny+ne,splus) = att(2*nx+2*ny+1:2*nx+2*ny+ne,splus);
            att_counter(counter_select,splus)=0;
            %%%%%%%%%%%%%%%%%%%%%%%%%%%%%%%
        end
        Filters.att(:,t,splus)          = att(:,splus);
        Filters.att_counter(:,t,splus)  = att_counter(:,splus);
    end
    
    % -- Kim-Nelson Collapsing Step for Covariance Matrix -- %
    for splus=1:h
        for st=1:h
            pai_st_splus            = PAI01_tt(st,splus)/PAItt(splus);
            Ptt_temp(:,:,splus,st)  = pai_st_splus*(Ptt_raw(:,:,st,splus) + (att(:,splus)-att_raw(:,st,splus))*(att(:,splus)-att_raw(:,st,splus))');
            pai_st_splus_counter            = PAI01_tt_counter(st,splus)/PAItt_counter(splus);
            Ptt_temp_counter(:,:,splus,st)  = pai_st_splus_counter*(Ptt_raw_counter(:,:,st,splus) + (att_counter(:,splus)-att_raw_counter(:,st,splus))*(att_counter(:,splus)-att_raw_counter(:,st,splus))');
        end
        Ptt(:,:,splus)              = sum(Ptt_temp(:,:,splus,:),4);
        Ptt_counter(:,:,splus)      = sum(Ptt_temp_counter(:,:,splus,:),4);
        if t == counter_t
            temp = 3:8;
            temp = temp(2+counter_e~=temp);  
            Ptt_counter(temp,:,splus) = 0;
            Ptt_counter(:,temp,splus) = 0;
            Ptt_raw_counter(2*nx+2*ny+1:2*nx+2*ny+ne,:,st,splus) = 0;
            Ptt_raw_counter(:,2*nx+2*ny+1:2*nx+2*ny+ne,st,splus) = 0;
        end
        Filters.Ptt(:,:,t,splus)    = Ptt(:,:,splus);
        Filters.Ptt_counter(:,:,t,splus)    = Ptt_counter(:,:,splus);
    end
end

% -- Sum Full Set of Likelihoods across Periods -- %
loglik=sum(Incr);
loglik_counter=sum(Incr_counter);
